"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = require("lodash");
var _coreHttpServer = require("@kbn/core-http-server");
var _coreHttpRouterServerInternal = require("@kbn/core-http-router-server-internal");
var _invalid_session_error = _interopRequireDefault(require("../errors/invalid_session_error"));
var _session_expired_error = _interopRequireDefault(require("../errors/session_expired_error"));
var _filter_auth_headers = _interopRequireDefault(require("../filter_auth_headers"));
var _missing_tenant_error = _interopRequireDefault(require("../errors/missing_tenant_error"));
var _path = _interopRequireDefault(require("path"));
var _multitenancy = require("../../../../../common/multitenancy");
/* eslint-disable @kbn/eslint/require-license-header */
/**
 *    Copyright 2018 floragunn GmbH

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at

 http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.
 */

class AuthType {
  constructor({
    searchGuardBackend,
    kibanaCore,
    config,
    logger,
    sessionStorageFactory,
    pluginDependencies,
    spacesService
  }) {
    /**
     * Called internally and checks for an AJAX request before
     * invoking the auth type's OnUnAuthenticated method.
     * @param request
     * @param response
     * @param toolkit
     * @param error
     * @returns {Promise<void|*>}
     * @private
     */
    (0, _defineProperty2.default)(this, "_handleUnAuthenticated", async (request, response, toolkit, error = null) => {
      // We don't have valid cookie credentials, but we may have an optional auth
      try {
        if (request.route.options.authRequired === 'optional') {
          return toolkit.next();
        }
      } catch (error) {
        this.logger.info('Could not read auth options for the path: ' + request.url.pathname);
      }
      if (request.headers) {
        // If the session has expired, we may receive ajax requests that can't handle a 302 redirect.
        // In this case, we trigger a 401 and let the interceptor handle the redirect on the client side.
        if (request.headers.accept && request.headers.accept.split(',').indexOf('application/json') > -1 || request.headers['content-type'] && request.headers['content-type'].indexOf('application/json') > -1) {
          this.debugLog('Not authenticated, detected AJAX request');
          const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
          if (request.route.path === '/api/core/capabilities') {
            return toolkit.notHandled();
          }
          return response.unauthorized({
            headers: {
              sg_redirectTo: await this.getRedirectTargetForUnauthenticated(request, error, true, sessionCookie)
            },
            body: {
              message: 'Session expired'
            }
          });
        }
      }
      return this.onUnAuthenticated(request, response, toolkit, error);
    });
    (0, _defineProperty2.default)(this, "onPostAuth", async (request, response, toolkit) => {
      if (request.route.path === '/api/core/capabilities') {
        const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
        if (sessionCookie.isAnonymousAuth) return toolkit.next();
        const authHeaders = await this.getAllAuthHeaders(request);
        if (authHeaders === false) {
          /*
          We need this redirect because Kibana calls the capabilities on our login page. The Kibana checks if there is the default space in the Kibana index.
          The problem is that the Kibana call is scoped to the current request. And the current request doesn't contain any credentials in the headers because the user hasn't been authenticated yet.
          As a result, the call fails with 401, and the user sees the Kibana error page instead of our login page.
          We flank this issue by redirecting the Kibana call to our route /api/v1/searchguard/kibana_capabilities where we serve some
          minimum amount of capabilities. We expect that Kibana fetches the capabilities again once the user logged in.
          */
          // The payload is passed together with the redirect despite of the undefined here
          return new _coreHttpServer.KibanaResponse(307, undefined, {
            headers: {
              location: this.basePath + '/api/v1/searchguard/kibana_capabilities'
            }
          });
        } else {
          // Update the request with auth headers in order to allow Kibana to check the default space.
          // Kibana page breaks if Kibana can't check the default space.
          const rawRequest = (0, _coreHttpRouterServerInternal.ensureRawRequest)(request);
          (0, _lodash.assign)(rawRequest.headers, authHeaders);
        }
      }
      return toolkit.next();
    });
    (0, _defineProperty2.default)(this, "checkAuth", async (request, response, toolkit) => {
      let sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
      try {
        sessionCookie = await this.getCookieWithCredentials(request);
      } catch (error) {
        return this._handleUnAuthenticated(request, response, toolkit, error);
      }
      if (sessionCookie.credentials) {
        const authHeaders = await this.getAllAuthHeaders(request, sessionCookie);
        if (!authHeaders) {
          this.logger.error(`An error occurred while computing auth headers, clearing session: No headers found in the session cookie`);
          await this.clear(request);
          return this._handleUnAuthenticated(request, response, toolkit);
        }
        const isMtEnabled = this.config.get('searchguard.multitenancy.enabled');
        if (!isMtEnabled && this.pluginDependencies.spaces) {
          await this.spacesService.createDefaultSpace({
            request: {
              headers: authHeaders
            }
          });
        }
        const rawRequest = (0, _coreHttpRouterServerInternal.ensureRawRequest)(request);
        (0, _lodash.assign)(rawRequest.headers, authHeaders);
        return toolkit.next();
      }
      return this._handleUnAuthenticated(request, response, toolkit);
    });
    this.searchGuardBackend = searchGuardBackend;
    this.config = config;
    this.kibanaCore = kibanaCore;
    this.logger = logger;
    this.sessionStorageFactory = sessionStorageFactory;
    this.pluginDependencies = pluginDependencies;
    this.spacesService = spacesService;
    this.basePath = kibanaCore.http.basePath.get();
    this.frontendBaseUrl = this.config.get('searchguard.frontend_base_url') || kibanaCore.http.basePath.publicBaseUrl;
    this.sgFrontendConfigId = this.config.get('searchguard.sg_frontend_config_id') || 'default';
    if (!this.frontendBaseUrl) {
      const serverInfo = kibanaCore.http.getServerInfo();
      this.frontendBaseUrl = serverInfo.protocol + '://' + serverInfo.hostname + ':' + serverInfo.port + '/' + kibanaCore.http.basePath.serverBasePath;
    }
    this.authDebugEnabled = this.config.get('searchguard.auth.debug');

    /**
     * The authType is saved in the auth cookie for later reference
     * @type {string}
     */
    this.type = null;

    /**
     * If a loginURL is defined, we can skip the auth selector page
     * if the customer only has one auth type enabled.
     * @type {string|null}
     */
    this.loginURL = null;

    /**
     * Tells the sessionPlugin whether or not to validate the number of tenants when authenticating
     * @type {boolean}
     */
    this.validateAvailableTenants = true;

    /**
     * The name of the header were we look for an authorization value.
     * This should most likely be set in the subclass depending on a config value.
     * @type {string}
     */
    this.authHeaderName = 'authorization';

    /**
     * Additional headers that should be passed as part as the authentication.
     * Do not use headers here that have an effect on which user is logged in.
     * @type {string[]}
     */
    this.allowedAdditionalAuthHeaders = ['sg_impersonate_as'];
  }
  async init() {
    this.setupRoutes();
  }

  /**
   * Returns the auth header needed for the Search Guard backend
   * @param session
   * @returns {*}
   */
  getAuthHeader(session) {
    if (session.credentials && session.credentials.authHeaderValue) {
      return {
        [this.authHeaderName]: session.credentials.authHeaderValue
      };
    }
    return false;
  }

  /**
   * Can be used by auth types that need to handle cases
   * where the credentials are passed together with the
   * request.
   * Example: JWT supports passing the bearer token per query parameter
   *
   * NB: Should NOT be used to detect pre-authenticated requests.
   * For those, we don't want to create a cookie.
   *
   * @param request
   * @returns {Promise<null>}
   */
  async detectCredentialsByRequest({
    request
  }) {
    return null;
  }

  /**
   * Checks if we have an authorization header.
   *
   * Pass the existing session credentials to compare with the authorization header.
   *
   * @param request
   * @param sessionCredentials
   * @returns {object|null} - credentials for the authentication
   */
  detectAuthHeaderCredentials(request, sessionCredentials = null) {
    if (request.headers[this.authHeaderName]) {
      const authHeaderValue = request.headers[this.authHeaderName];

      // If we have sessionCredentials AND auth headers we need to check if they are the same.
      if (sessionCredentials !== null && sessionCredentials.authHeaderValue === authHeaderValue) {
        // The auth header credentials are the same as those in the session,
        // no need to return new credentials so we're just returning null here
        return null;
      }
      return {
        authHeaderValue: authHeaderValue
      };
    }
    return null;
  }
  async getRedirectTargetForUnauthenticated() {
    throw new Error('The getRedirectTargetForUnauthenticated method must be implemented by the sub class');
  }
  async authenticate(credentials, options = {}, additionalAuthHeaders = {}) {
    try {
      this.debugLog('Authenticating using ' + credentials);
      credentials.frontend_base_url = this.frontendBaseUrl;
      credentials.config_id = this.sgFrontendConfigId;
      const sessionResponse = await this.searchGuardBackend.authenticateWithSession(credentials);
      const sessionCredentials = {
        authHeaderValue: 'Bearer ' + sessionResponse.token
      };
      this.debugLog('Token ' + sessionCredentials.authHeaderValue);
      const user = await this.searchGuardBackend.authenticateWithHeader(this.authHeaderName, sessionCredentials.authHeaderValue, additionalAuthHeaders);
      const session = {
        username: user.username,
        // The session token
        credentials: sessionCredentials,
        authType: this.type,
        authTypeId: credentials.id
      };
      return {
        session,
        user,
        redirectUri: sessionResponse.redirect_uri
      };
    } catch (error) {
      throw error;
    }
  }
  async onUnAuthenticated(request, response, toolkit, error = null) {
    const redirectTo = await this.getRedirectTargetForUnauthenticated(request, error);
    return response.redirected({
      headers: {
        location: `${redirectTo}`
      }
    });
  }

  /**
   * A helper for generating the correct nextUrl.
   * Spaces manipulates the URL for non default
   * spaces, and that change is not reflected
   * in request.url.pathname
   * @param request
   * @returns {string}
   */
  getNextUrl(request) {
    return _path.default.posix.join(this.basePath, request.url.pathname) + request.url.search;
  }
  setupRoutes() {
    throw new Error('The getAuthHeader method must be implemented by the sub class');
  }
  async getCookieWithCredentials(request) {
    const authHeaderCredentials = await this.detectCredentialsByRequest({
      request
    });
    if (authHeaderCredentials) {
      try {
        this.debugLog('Got auth header credentials, trying to authenticate');
        const {
          session
        } = await this.handleAuthenticate(request, authHeaderCredentials);
        return session;
      } catch (error) {
        this.logger.error(`Got auth header credentials, but authentication failed: ${error.stack}`);
        // Fall through
      }
    }

    let sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    if (sessionCookie.credentials) {
      try {
        return await this.validateSessionCookie(request, sessionCookie);
      } catch (error) {
        // We can return early here. Even if we have valid request headers,
        // the cookie would have been updated in the validator.
        // Logging this as info since it isn't really an error, but just a part of the flow.
        this.logger.info(`Got credentials, but the validation failed: ${error.stack}`);
        // Fall through
      }
    }

    // No (valid) cookie, we need to check for headers

    return sessionCookie;
  }
  /**
   * If a session auth cookie exists, the sessionValidator is called to validate the content.
   * If the cookie isn't valid, an error will be thrown
   * @param server
   * @returns {validate}
   */
  async validateSessionCookie(request, sessionCookie) {
    if (sessionCookie.authType !== this.type) {
      await this.clear(request);
      throw new _invalid_session_error.default('Invalid cookie');
    }
    const checkTokenExpirationTime = Date.now();
    if (!sessionCookie.checkTokenExpirationTime || checkTokenExpirationTime - sessionCookie.checkTokenExpirationTime > 15000) {
      try {
        const authHeader = this.getAuthHeader(sessionCookie);
        const authInfoResponse = await this.searchGuardBackend.authinfo(authHeader || request.headers);
        sessionCookie.checkTokenExpirationTime = checkTokenExpirationTime;
        await this.sessionStorageFactory.asScoped(request).set(sessionCookie);
        if (authInfoResponse.user_name !== sessionCookie.username) {
          throw new Error('We have a different user in the cookie. Most likely because of anonymous auth.');
        }
      } catch (error) {
        await this.clear(request);
        throw new _session_expired_error.default('Session expired');
      }
    }
    return sessionCookie;
  }

  /**
   * Get all auth headers based on the current request.
   *
   * @param request
   * @param sessionCookie
   * @returns {Promise<boolean|*>}
   */
  async getAllAuthHeaders(request, sessionCookie = null) {
    if (!sessionCookie) {
      try {
        sessionCookie = await this.getCookieWithCredentials(request);
      } catch (error) {
        this.logger.info(`Getting all auth headers failed: ${error.stack}`);
      }
    }
    if (!sessionCookie || !sessionCookie.credentials) {
      return false;
    }
    const authHeader = this.getAuthHeader(sessionCookie);
    if (authHeader !== false) {
      this.addAdditionalAuthHeaders(request, authHeader, sessionCookie);
      return authHeader;
    }
    return false;
  }

  /**
   * @deprecated
   *
   * Method for adding additional auth type specific authentication headers.
   * Override this in the auth type for type specific headers.
   *
   * NB: Remember to call the super method if you do.
   *
   * @param request
   * @param authHeader
   * @param session
   */
  addAdditionalAuthHeaders(request, authHeader, session) {
    if (session.additionalAuthHeaders) {
      for (const header in session.additionalAuthHeaders) {
        if (session.additionalAuthHeaders.hasOwnProperty(header)) {
          authHeader[header] = session.additionalAuthHeaders[header];
        }
      }
    }
  }
  debugLog(message, label = this.type) {
    if (this.authDebugEnabled) {
      try {
        if (typeof message !== 'string') {
          message = JSON.stringify(message);
        }
        this.logger.info(`${label} ${message}`);
      } catch (error) {
        this.logger.error(`Error in debug log: ${error.stack}`);
      }
    }
  }

  /**
   * Tries to authenticate a user. If multitenancy is enabled, we also try to validate that the
   * user has at least one valid tenant
   * @param {object} request
   * @param {object} headers
   * @param {object} credentials
   * @returns {Promise<*>}
   */
  async handleAuthenticate(request, credentials, options = {}) {
    try {
      const additionalAuthHeaders = (0, _filter_auth_headers.default)(request.headers, this.allowedAdditionalAuthHeaders);
      // authResponse is an object with .session and .user
      const authResponse = await this.authenticate(credentials, options, additionalAuthHeaders);
      return this._handleAuthResponse(request, credentials, authResponse, additionalAuthHeaders);
    } catch (error) {
      // Make sure we clear any existing cookies if something went wrong
      this.clear(request, true);
      throw error;
    }
  }

  /**
   * Normalized response after an authentication
   * @param credentials
   * @param authResponse
   * @returns {*}
   * @private
   */
  async _handleAuthResponse(request, credentials, authResponse, additionalAuthHeaders = {}) {
    // Make sure the user has a tenant that they can use

    if (this.validateAvailableTenants && this.config.get('searchguard.multitenancy.enabled')) {
      let userTenantInfo;
      let allTenants = {};
      try {
        userTenantInfo = await this.searchGuardBackend.getUserTenantInfo({
          authorization: authResponse.session.credentials.authHeaderValue
        });
        userTenantInfo = this.searchGuardBackend.removeNonExistingReadOnlyTenants(userTenantInfo);
        allTenants = this.searchGuardBackend.convertUserTenantsToRecord(userTenantInfo.data.tenants);
      } catch (error) {
        this.logger.info(`Could not retrieve the user tenants`);
      }
      if (!userTenantInfo || Object.keys(allTenants).length === 0) {
        throw new _missing_tenant_error.default('No tenant available for this user, please contact your system administrator.');
      }
    }

    // If we used any additional auth headers when authenticating, we need to store them in the session
    /* @todo Was used with sg_impersonate_as
    authResponse.session.additionalAuthHeaders = null;
    if (Object.keys(additionalAuthHeaders).length) {
      authResponse.session.additionalAuthHeaders = additionalAuthHeaders;
    }
      */

    const cookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    authResponse.session = {
      ...cookie,
      ...authResponse.session
    };
    await this.sessionStorageFactory.asScoped(request).set(authResponse.session);
    return authResponse;
  }
  async logout({
    request,
    response
  }) {
    await this.clear(request, true);
    return response.ok({
      body: {
        authType: this.type,
        // @todo Use the kibana_url from the config?
        redirectURL: this.basePath + '/searchguard/login?type=' + this.type + 'Logout'
      }
    });
  }

  /**
   * Remove the credentials from the session cookie
   */
  async clear(request, explicitlyRemoveAuthType = false) {
    const sessionCookie = (await this.sessionStorageFactory.asScoped(request).get()) || {};
    const authHeaders = this.getAuthHeader(sessionCookie);
    // @todo Consider refactoring anything auth related into an "auth" property.
    delete sessionCookie.credentials;
    delete sessionCookie.username;
    if (explicitlyRemoveAuthType) {
      delete sessionCookie.authType;
      delete sessionCookie.authTypeId;
    }
    delete sessionCookie.additionalAuthHeaders;

    // Only try to delete the session if we really have auth headers
    if (authHeaders) {
      try {
        await this.searchGuardBackend.logoutSession(authHeaders);
      } catch (error) {
        this.logger.error(`Failed to delete the session token: ${error.stack}`);
      }
    }
    return await this.sessionStorageFactory.asScoped(request).set(sessionCookie);
  }
}
exports.default = AuthType;
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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